/**
 *  @file    PicturePointColor
 *  @author  moca
 *  @version 1.0.1 2022/08/29
 */

/*:ja
 * @plugindesc マウスカーソルがピクチャ上にある時、その色を変数に代入します
 * また、ピクチャ上にある指定座標の色も取得できます
 * @author moca
 * 
 * @param picturePointStopSwitch
 * @text ピクチャポイントカラー有効/無効スイッチ
 * @desc 指定したスイッチがONならカラーマップの色を取得します
 * @default 1
 * @type switch
 *
 * @help ▼----マウスカーソルがピクチャ上にある時、その色を変数に代入します----▼
 *
 * ●ピクチャの色を16進数で取得する
 * 
 * 　PPC_GET_RGB [ピクチャ番号 (または、\v[変数番号])] [変数番号(または、\v[変数番号])]
 *
 * 例1：マウス座標にあるピクチャ番号1の色を変数番号10に代入する
 * 　PPC_GET_RGB 1 10 
 *
 * 例2：マウス座標にあるピクチャ番号を変数1で指定し、色を変数2で指定した変数に代入する
 * 　PPC_GET_RGB \v[1] \v[2]
 *
 * スクリプト呼び出し
 * $gameScreen.getPicturePointerColorRGB([ピクチャ番号], [変数番号]);
 *
 *
 * ●ピクチャの色を10進数で取得する 
 *
 * 　PPC_GET_RGB_DEC [ピクチャ番号(または、\v[変数番号])] [変数番号(または、\v[変数番号])]
 *
 * 例1：マウス座標にあるピクチャ番号1の色を10進数に変換して変数番号10に代入する)
 * 　PPC_GET_RGB_DEC 1 10
 *
 * 例2：マウス座標にあるピクチャ番号を変数1で指定し、色を10進数に変換して変数2で指定した変数に代入する)
 * 　PPC_GET_RGB_DEC \v[1] \v[2]
 *
 * スクリプト呼び出し
 *  $gameScreen.getPicturePointerColorRGB_DEC([ピクチャ番号], [変数番号]);
 *
 *
 *
 *▼----指定した座標の色を変数に代入します----▼
 *
 * ●画面座標、または変数で指定した座標のピクチャの色を16進数で取得する
 * 　PPC_GET_ANYPOSITIONRGB [ピクチャ番号(または、\v[変数番号])] [変数番号(または、\v[変数番号])] [画面X座標 (または、\v[変数番号])] [画面Y座標 (または、\v[変数番号])]
 *
 * 例1：画面座標(100,200)にあるピクチャ番号1の色を変数番号10に代入する
 * 　PPC_GET_ANYPOSITIONRGB 1 10 100 200
 *
 * 例2：変数4と5で座標を指定し、ピクチャ番号を変数1で指定し、変数10で指定した変数に代入する
 * 　PPC_GET_ANYPOSITIONRGB \v[1] \v[10] \v[4] \v[5]
 *
 * スクリプト呼び出し
 *  $gameScreen.getPicturePointerColorAnyPositionRGB( [ピクチャ番号] [変数番号] [X座標] [Y座標] )
 *
 *
 * ●画面座標、または変数で指定した座標のピクチャの色を10進数で取得する
 * 　PPC_GET_ANYPOSITIONRGB_DEC [ピクチャ番号(または、\v[変数番号])] [変数番号(または、\v[変数番号])] [画面X座標 (または、\v[変数番号])] [画面Y座標 (または、\v[変数番号])]
 *
 * 例1：画面座標(100,200)にあるピクチャ番号1の色を10進数に変換して変数番号10に代入する
 * 　PPC_GET_ANYPOSITIONRGB_DEC 1 10 100 200
 *
 * 例2：変数4と5で座標を指定し、ピクチャ番号を変数1で指定し、色を10進数に変換して変数15で指定した変数に代入する
 * 　PPC_GET_ANYPOSITIONRGB_DEC \v[1] \v[15] \v[4] \v[5]
 * 
 * スクリプト呼び出し
 *  $gameScreen.getPicturePointerColorAnyPositionRGB_DEC( [ピクチャ番号] [変数番号] [X座標] [Y座標] )
 *
 *
 * ## バージョン履歴
 * 2024/03/01 カラーマップ取得　有効/無効 スイッチを追加　追加:trs
 * 2023/12/15 ピクチャの色を画面座標から取得する機能を追加 追加:trs
 * 2022/08/29 ピクチャの色を10進数で取得する機能を追加
 * 2022/08/24 1.0.0 初版
 * 
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
*/

(function () {
	 //-------↓add start 2024/03/01 trs------
	var parameters = PluginManager.parameters('PicturePointColor_EX_v2');
	var params = [];
	
	params.enable_switch = Number ( parameters['picturePointStopSwitch'] || 1 );
	
	 //-------↑add end 2024/03/01 trs------
    'use strict';

    /**
     * ゲーム内変数を取得する
     * @param {String} text \V[n]形式の制御文字
     * @returns {number} ゲーム内変数
     */
    var getUsingVariable = function (text) {
        text = text.replace(/\\/g, '\x1b');
        text = text.replace(/\x1b\x1b/g, '\\');
        text = text.replace(/\\/g, '\x1b');
        text = text.replace(/\x1b\x1b/g, '\\');
        text = text.replace(/\x1bV\[(\d+)\]/gi, function () {
            return $gameVariables.value(parseInt(arguments[1]));
        }.bind(this));
        text = text.replace(/\x1bV\[(\d+)\]/gi, function () {
            return $gameVariables.value(parseInt(arguments[1]));
        }.bind(this));
        return parseInt(text);
    };

    // プラグインコマンド
    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function (command, args) {
        _Game_Interpreter_pluginCommand.apply(this, arguments);

        if (command.toLowerCase() === "ppc_get_rgb") {
            if (args.length < 2) {
                throw "プラグインコマンド PPC_GET_RGB の引数が不足しています"
            }
            else {
                $gameScreen.getPicturePointColorRGB(getUsingVariable(args[0]), getUsingVariable(args[1]));
            }
        }
        else if (command.toLowerCase() === "ppc_get_rgb_dec") {
            if (args.length < 2) {
                throw "プラグインコマンド PPC_GET_RGB_DEC の引数が不足しています"
            }
            else {
                $gameScreen.getPicturePointColorRGB_DEC(getUsingVariable(args[0]), getUsingVariable(args[1]));
            }
        }
        
        //-------↓add start 2023/12/15 trs------
        else if (command.toLowerCase() === "ppc_get_anypositionrgb") {
            if (args.length < 4) {
                throw "プラグインコマンド PPC_GET_AnyPositionRGB の引数が不足しています"
            }
            else {
                 $gameScreen.getPicturePointerColorAnyPositionRGB(getUsingVariable(args[0]), getUsingVariable(args[1]), getUsingVariable(args[2]), getUsingVariable(args[3]) );
            }
        }
        else if (command.toLowerCase() === "ppc_get_anypositionrgb_dec") {
            if (args.length < 4) {
                throw "プラグインコマンド PPC_GET_AnyPositionRGB_DEC の引数が不足しています"
            }
            else {
                $gameScreen.getPicturePointerColorAnyPositionRGB_DEC(getUsingVariable(args[0]), getUsingVariable(args[1]), getUsingVariable(args[2]), getUsingVariable(args[3]) );
            }
        }
        //-------↑add end 2023/12/15 trs------
    };

    Game_Screen.prototype._getPicturePointColor = function (pictureId) {
        let pic = this.picture(pictureId);
        let result = -1;
        if (pic !== undefined) {
            let sprite = new Sprite_Picture(pictureId);

            let w = sprite.bitmap.width;
            let h = sprite.bitmap.height;
            let scaleX = pic.scaleX() / 100;
            let scaleY = pic.scaleY() / 100;

            let sx = TouchInput.x - pic.x();
            let sy = TouchInput.y - pic.y();

            // ピクチャの基点が中央ならずらす
            if (pic.origin() === 1) {
                sx += scaleX * w / 2;
                sy += scaleY * h / 2;
            }
            let x = sx * (w / (w * scaleX));
            let y = sy * (h / (h * scaleY));

            let color = sprite.bitmap.getPixel(x, y);
            // ピクチャの外だった場合は-1を返す
            if (x < 0 || y < 0 || x > w || y > h) {
                color = -1;
            }

            result = color;
        }

        return result;
    };
	
	//---↓add start 2023/12/15 trs ---
	Game_Screen.prototype._getPicturePointerColorAnyPositionRGB = function (pictureId, _x, _y) {
        let pic = this.picture(pictureId);
        let result = -1;
        if (pic !== undefined) {
            let sprite = new Sprite_Picture(pictureId);

            let w = sprite.bitmap.width;
            let h = sprite.bitmap.height;
            let scaleX = pic.scaleX() / 100;
            let scaleY = pic.scaleY() / 100;

            let sx = _x - pic.x();
            let sy = _y - pic.y();

            // ピクチャの基点が中央ならずらす
            if (pic.origin() === 1) {
                sx += scaleX * w / 2;
                sy += scaleY * h / 2;
            }
            let x = sx * (w / (w * scaleX));
            let y = sy * (h / (h * scaleY));

            let color = sprite.bitmap.getPixel(x, y);
            // ピクチャの外だった場合は-1を返す
            if (x < 0 || y < 0 || x > w || y > h) {
                color = -1;
            }

            result = color;
        }

        return result;
    };
	
	//指定
	Game_Screen.prototype.getPicturePointerColorAnyPositionRGB = function (pictureId, valueId, _posX, _posY) {
        let color = $gameScreen._getPicturePointerColorAnyPositionRGB(pictureId, _posX, _posY);
        
        //---↓del start 2024/03/01 trs---
        //$gameVariables.setValue(valueId, color);
        //---↑del end 2024/03/01 trs---
        
        //---↓add start 2024/03/01 trs---
        if( $gameSwitches.value(params.enable_switch) === true ){
        	$gameVariables.setValue(valueId, color);
        }
        else{
        	$gameVariables.setValue(valueId, -1);
        }
        //---↑add end 2024/03/01 trs---
    };

    Game_Screen.prototype.getPicturePointerColorAnyPositionRGB_DEC = function (pictureId, valueId, _posX, _posY) {
        let color = $gameScreen._getPicturePointerColorAnyPositionRGB(pictureId, _posX, _posY);
        if (color !== -1) {
            color = parseInt(color.replace('#', ''), 16);
        }
        
        //---↓del start 2024/03/01 trs---
        //$gameVariables.setValue(valueId, color);
        //---↑del end 2024/03/01 trs---
        
        //---↓add start 2024/03/01 trs---
        if( $gameSwitches.value(params.enable_switch) === true ){
        	$gameVariables.setValue(valueId, color);
        }
        else{
        	$gameVariables.setValue(valueId, -1);
        }
        //---↑add end 2024/03/01 trs---
        
    };
	
	//----↑add end 2023/12/15 trs----
	
    Game_Screen.prototype.getPicturePointColorRGB = function (pictureId, valueId) {
        let color = $gameScreen._getPicturePointColor(pictureId);
        //---↓del start 2024/03/01 trs---
        //$gameVariables.setValue(valueId, color);
        //---↑del end 2024/03/01 trs---
        
        //---↓add start 2024/03/01 trs---
        if( $gameSwitches.value(params.enable_switch) === true ){
        	$gameVariables.setValue(valueId, color);
        }
        else{
        	$gameVariables.setValue(valueId, -1);
        }
        //---↑add end 2024/03/01 trs---
    };

    Game_Screen.prototype.getPicturePointColorRGB_DEC = function (pictureId, valueId) {
        let color = $gameScreen._getPicturePointColor(pictureId);
        if (color !== -1) {
            color = parseInt(color.replace('#', ''), 16);
        }
        //---↓del start 2024/03/01 trs---
        //$gameVariables.setValue(valueId, color);
        //---↑del end 2024/03/01 trs---
        
        //---↓add start 2024/03/01 trs---
        if( $gameSwitches.value(params.enable_switch) === true ){
        	$gameVariables.setValue(valueId, color);
        }
        else{
        	$gameVariables.setValue(valueId, -1);
        }
        //---↑add end 2024/03/01 trs---
    };

    TouchInput._onMouseMove = function (event) {
        var x = Graphics.pageToCanvasX(event.pageX);
        var y = Graphics.pageToCanvasY(event.pageY);
        this._onMove(x, y);
    };
})();